#!hs2
#!load hamster.hsm
#!load hstrings.hsm


########################################################################
# Script     : hsc-convert.hsc
# Description: Script, which converts an old-style Hamster-script into
#              the new-style scripts of Hamster Vr. 1.3.17 and later.
# Maintainer : Juergen Haible <juergen.haible@t-online.de>
# Version    : 2000-05-02
########################################################################


# initialize global vars
var( $CRLF, $RESTARTREPLACE, $MSGCAPTION )
$CRLF = chr(13) + chr(10)
$RESTARTREPLACE = "Goto( Restart )"
$MSGCAPTION = "Conversion of Hamster-script"

# select script to be converted
var( $ScriptFile, $ScriptList )
$ScriptList = ListAlloc
ListFiles( $ScriptList, "*.hsc" )
if( ListCount($ScriptList) = 0 )
   ErrorBox( "No scripts found in current directory!" )
   quit
endif

$ScriptFile = InputBox( _
   "Available scripts:" + $CRLF + BuildNewPars( $ScriptList ) _
   + $CRLF + $CRLF + "Enter script-file to be converted:", _
   $MSGCAPTION, "" )

if( $ScriptFile = "" )
   quit
endif

if( ! FileExists($ScriptFile) )
   ErrorBox( "Script '" + $ScriptFile + "' not found!" )
   quit
endif

# convert script
ConvertOldHscFile( $ScriptFile, $ScriptFile )

InfoBox( "Script '" + $ScriptFile + "' was successfully " _
        + "converted to new format." )

quit

########################################################################

sub ConvertOldHscFile( $OldFile, $NewFile )
# converts an old script-file into a new one

   # load and check old script
   var( $OldList )
   $OldList = ListAlloc
   if( ListLoad( $OldList, $OldFile ) <> 0 )
      ErrorBox( "Script '" + $OldFile + "' could not be loaded!" )
      quit
   endif
   if( ListGet( $OldList, 0 ) = "#!hs2" )
      ErrorBox( "Script '" + $OldFile + "' is already converted!" )
      quit
   endif

   # backup old script
   if( FileCopy( $OldFile, $OldFile + "-old" ) = 0 )
      InfoBox( "NOTE: Old script was backed up as '" _
               + $OldFile + "-old'." )
   else
      ErrorBox( "Old script could not be backed up as '" _
                + $OldFile + "-old'." )
      quit
   endif

   # create and prepare new script
   var( $NewList, $RestartIndex, $RestartNeeded )
   $NewList = ListAlloc
   ListAdd( $NewList, "#!hs2" )
   ListAdd( $NewList, "#!load hamster.hsm" )
   ListAdd( $NewList, "" )
   $RestartIndex  = ListAdd( $NewList, "Label( Restart )" )
   $RestartNeeded = False

   # convert old script line by line
   var( $LineNo, $Line )
   $LineNo = 0

   while( $LineNo < ListCount( $OldList ) )

      $Line = ListGet( $OldList, $LineNo )
      print( "OLD: ", $Line )

      $Line = ConvertOldHscLine( $Line )
      print( "NEW: ", $Line )
      ListAdd( $NewList, $Line )

      if( $Line = $RESTARTREPLACE )
         $RestartNeeded = True
      endif

      inc( $LineNo )

   endwhile

   # remove restart-label if it's not needed
   if( !$RestartNeeded )
      ListDelete( $NewList, $RestartIndex )
   endif

   # save new script
   if( ListSave( $NewList, $NewFile )<>0 )
      ErrorBox( "New script '+$NewFile+"' could not be saved!" )
      quit
   endif

   # clean up
   ListFree( $OldList )
   ListFree( $NewList )

endsub

########################################################################

sub ConvertOldHscLine( $Line )
# converts an old .hsc-line into the new format

   var( $p, $Old, $New, $Par, $ParList, $s )

   $Line = TrimWSP( $Line ) # remove whitespace

   # leave empty lines and comments unchanged
   if( ($Line="") || (copy($Line,1,1)="#") )
      return( $line )
   endif

   # separate command from parameters
   $p = PosWSP( $Line )
   if( $p = 0 )
      $Old = $Line
      $Par = ""
   else
      $Old = copy( $Line, 1, $p-1 )
      $Par = TrimWSP( copy( $Line, $p+1 ) )
   endif
   $Old = lowercase($Old)

   # split WSP-separated parameters into list-entries
   $ParList = ListAlloc
   SplitPars( $Par, $ParList )

   # convert old commands into new commands

   if( $Old = "call" )
      $New = "XXX # " + $Line  # XXX forces error when executed
      $Par = ""
      $Old = ""
      WarnBox( "NOTE: A 'call'-line could not be converted. " _
               + "It was commented out and marked with XXX instead:" _
               + $CRLF + $CRLF + $New )
   endif

   if( $Old = "fetchmail" )
      $New = "HamFetchMail"
      $s = ListGet($ParList,4) # separate "DestUser,FilterSection"
      $p = Pos( ",", $s )
      if( $p > 0 )
         ListAdd( $ParList, chr(34) + copy($s,$p+1) )
         ListSet( $ParList, 4, copy($s,1,$p-1) + chr(34) )
      endif
      $Par = BuildNewPars( $ParList )
      $Old = ""
   endif

   if( $Old = "mail.pull" )
      $New = "HamMailExchange"
      $Par = BuildNewPars( $ParList )
      $Old = ""
   endif

   if( $Old = "message" )
      $New = "HamMessage"
      $s = int( ListGet( $ParList, 0 ) )
      $s = icase( $s, _
                  $HAM_MSG_HAMSTER_EXIT,    "$HAM_MSG_HAMSTER_EXIT", _
                  $HAM_MSG_RESET_COUNTERS,  "$HAM_MSG_RESET_COUNTERS", _
                  $HAM_MSG_LOCALNNTP_ONOFF, "$HAM_MSG_LOCALNNTP_ONOFF", _
                  $HAM_MSG_LOCALPOP3_ONOFF, "$HAM_MSG_LOCALPOP3_ONOFF", _
                  $HAM_MSG_LOCALSMTP_ONOFF, "$HAM_MSG_LOCALSMTP_ONOFF", _
                  $HAM_MSG_SHOWWINDOW,      "$HAM_MSG_SHOWWINDOW", _
                  $HAM_MSG_SHOWICON,        "$HAM_MSG_SHOWICON", _
                  else,                     str( $s ) _
                )
      ListSet( $ParList, 0, $s )
      if( ListCount($ParList)<2 )
         ListAdd( $ParList, "0" )
      endif
      $Par = BuildNewPars( $ParList )
      $Old = ""
   endif

   if( $Old = "msgbox" )
      $New = "MsgBox"
      ListAdd( $ParList, chr(34)+"Hamster"+chr(34) )
      ListAdd( $ParList, "0x40" ) # info-style
      $Par = BuildNewPars( $ParList )
      $Old = ""
   endif

   if( $Old = "news.post" )
      $New = "HamNewsPost"
      $Par = BuildNewPars( $ParList )
      $Old = ""
   endif

   if( $Old = "news.pull" )
      $New = "HamNewsPull"
      $Par = BuildNewPars( $ParList )
      $Old = ""
   endif

   if( $Old = "news.purge" )
      $New = "HamPurge"
      $Par = BuildNewPars( $ParList )
      $Old = ""
   endif

   if( $Old = "news.rebuildgloballists" )
      $New = "HamRebuildGlobalLists"
      $Par = ""
      $Old = ""
   endif

   if( $Old = "news.rebuildhistory" )
      $New = "HamRebuildHistory"
      $Par = ""
      $Old = ""
   endif

   if( $Old = "quit" )
      $New = "Quit"
      $Par = ""
      $Old = ""
   endif

   if( $Old = "ras.dial" )
      $New = "HamRasDial"
      $Par = BuildNewPars( $ParList )
      $Old = ""
   endif

   if( $Old = "ras.hangup" )
      $New = "HamRasHangup"
      $Par = BuildNewPars( $ParList )
      $Old = ""
   endif

   if( $Old = "restart" )
      $New = $RESTARTREPLACE
      $Par = ""
      $Old = ""
   endif

   if( $Old = "sendmail" )
      $New = "HamSendMail"
      $Par = BuildNewPars( $ParList )
      $Old = ""
   endif

   if( $Old = "setlogin" )
      $New = "HamSetLogin"
      $Par = BuildNewPars( $ParList )
      $Old = ""
   endif

   if( ($Old="start.wait")||($Old="start.nowait") )
      $New = "Execute"
      ListAdd( $ParList, "HamPath" ) # <workdir>
      ListAdd( $ParList, 1         ) # <show>
      ListAdd( $ParList, iif($Old="start.wait","True","False") ) # <wait>
      $Par = BuildNewPars( $ParList )
      $Old = ""
   endif

   if( $Old = "wait.delay" )
      $New = "Sleep"
      $Par = BuildNewPars( $ParList )
      $Old = ""
   endif

   if( $Old = "wait.idle" )
      $New = "HamWaitIdle"
      $Par = ""
      $Old = ""
   endif

   if( $Old = "wait.until" )
      $New = "XXX # " + $Line  # XXX forces error when executed
      $Par = ""
      $Old = ""
      WarnBox( "NOTE: A 'wait.until'-line could not be converted. " _
               + "It was commented out and marked with XXX instead:" _
               + $CRLF + $CRLF + $New )
   endif

   ListFree( $ParList )

   if( $Old <> "" )
      ErrorBox( "Unknown command in line: " + $Line )
      quit
   endif

   # return command-line in new format
   if( $Par = "" )
      return( $New )
   else
      return( $New + "( " + $Par + " )" )
   endif

endsub

########################################################################

sub SplitPars( $Pars, $ParList )
# splits WSP-separated parameters into list-entries

   var( $p )

   ListClear( $ParList )
   $Pars = TrimWSP( $Pars )

   while( $Pars <> "" )
      break( copy($Pars,1,1)="#" ) # rest of line is comment

      if( copy($Pars,1,1)=chr(34) )
         $p = Pos( chr(34), $Pars, 2 )
      else
         $p = PosWSP($Pars) - 1
      endif

      if( $p <= 0 )
         ListAdd( $ParList, $Pars )
         $Pars = ""
      else
         ListAdd( $ParList, copy( $Pars, 1, $p ) )
         $Pars = TrimWSP( copy( $Pars, $p+1 ) )
      endif
   endwhile

endsub

########################################################################

sub BuildNewPars( $ParList )
# builds comma-separated parameter-list based on list-entries

   var( $Par, $i )

   $Par = ""
   $i = 0

   while( $i < ListCount( $ParList ) )
      $Par = $Par + iif( $i=0, "", ", " )
      $Par = $Par + ListGet( $ParList, $i )
      inc( $i )
   endwhile

   return( $Par )

endsub

########################################################################

sub ErrorBox( $Msg )
   MsgBox( $Msg, $MSGCAPTION, 0x10 )
endsub

sub WarnBox( $Msg )
   MsgBox( $Msg, $MSGCAPTION, 0x30 )
endsub

sub InfoBox( $Msg )
   MsgBox( $Msg, $MSGCAPTION, 0x40 )
endsub

########################################################################
